%% MARS-PHOBOS-DEIMOS TERNARY SYSTEM

%
% Planar Bicircular Restricted 4-Body Problem (PBCR4BP)
%
%DESCRIPTION:
% Questo codice permette di visualizzare le migliori traiettorie di escape
% assistito e non nel caso specifico del sistema ternario Marte-Phobos-Deimos
%
%AUTHOR(s):
% Alessio Patti, MSc Politecnico di Torino 2023,
% "Minimum-Propellant Direct, Assisted, and Slingshot Return Trajectories
% from Mars, Jupiter, and their moons."
% alessio.patti@studenti.polito.it

clc
clear all
close all
format short

%% DATI STRUCT

% Constants and Parameters Mars-Phobos-Deimos

% m1 --> Marte1
% m2 --> Phobos
% m3 --> Deimos

% N.B. la funzione PBR4BP prende in input data.mu e data.m3_dim per
% implementare il problema dei 4 corpi planare, per cui se si vogliono
% modificare i parametri gravitazionali dei corpi in gioco ricordare che
% bisogno modificare nella funzinoe PBR4BP data.mu2_dim per Phobos e
% data.m3_dim per Deimos!

b = 0;

for d=1:0.25:3                                                             % ciclo sulla distanza tra primario e terziario RHO13 (la distana tra secondario e terziario sarà rho13-1)

    for c=0:1:8                                                                % ciclo sulla massa del terziario M3

        for a=0:1:7                                                            % ciclo sul parametro gravitazionale binario MU12

            b=b+1;

            MU12 = 10.^(a);
            M3 = 10^(c);
            RHO13 = d;

            data.m1_dim = 6.39e23;                                                        % [Kg]
            data.m2_dim = 1.08e16;                                                        % [Kg]
            data.m3_dim = 2e15;                                                           % [Kg]

            data.mu1_dim = 42828;                                                         % [km^3/s^2]
            data.mu2_dim = 0.0007112;                                                     % [km^3/s^2]
            data.mu3_dim = 0.0000985;                                                     % [km^3/s^2]

            data.mu = (data.mu2_dim *MU12 ) / (data.mu1_dim + data.mu2_dim);              % [/]

            data.M = data.m1_dim + data.m2_dim;                                           % [/]
            data.m1_adim = 1 - data.mu;                                                   % [/]
            data.m2_adim = data.mu;                                                       % [/]
            data.m3_adim = (data.m3_dim *M3) / (data.M);                                  % [/]

            data.Raggio_12 = 9378;                                                      % [km]
            data.Raggio_13 =  23459;                                                    % Possiamo approssimarla come la distanza tra Deimos e il baricentro del sistema Marte-Phobos [km]
            data.Raggio_Marte = 3389;                                                   % [Km]
            data.Raggio_Phobos = 11.267;                                                % [Km]
            data.Raggio_Deimos = 6.2;

            data.th0P = 0*pi/180;
            data.th0D = 0*pi/180; % nel flyby combinato la reale posizione Deimos è definita da (thetaD - thetaP) % 61

            data.omega_Phobos = sqrt((data.mu1_dim + data.mu2_dim) / data.Raggio_12^3); % [rad/s]

            data.h = 400;                                                                % [Km]
            data.DU_12 = 9378;                                                           % [km]
            data.DU_13 = 23459;                                                          % [km]

            data.DU_12_adim = data.DU_12 / data.DU_12;                                   % [/]
            %  data.DU_13_adim = (data.DU_13 / data.DU_12);                                 % [/]
            data.DU_13_adim = RHO13;                                                     % [/]

            data.VU_2 = sqrt( (data.mu1_dim + data.mu2_dim) / data.Raggio_12);           % [Km/s]
            data.omega_2 = sqrt( (data.mu1_dim + data.mu2_dim) / data.Raggio_12^3);      % [rad/s]

            data.VU_3 = sqrt( (data.mu1_dim + data.mu2_dim) / data.Raggio_13);
            data.omega_3 = sqrt( (data.mu1_dim + data.mu2_dim) / data.Raggio_13^3);  % Questa è la velocità angolare del terzo corpo in rad/s ma nella funzione PBR4BP và inserita in modo adimensinionale quindi bisogna dividere per la omega di riferimento che è quella di Phobos


            disp('Indice =')
            disp(b)
            disp('MU12 =')
            disp(data.mu)
            disp('M3 =')
            disp(data.m3_adim)
            disp('RHO13 = ')
            disp(data.DU_13_adim)



            %% DATI

            n_iter = 2000;

            massa_Marte = 6.39e23;                                                      % [Kg]
            massa_Phobos = 1.08e16;                                                     % [Kg]
            massa_Deimos = 2e15;                                                        % [Kg]
            massa_Sole = 1.989e30;                                                      % [Kg]

            mu_Marte = 42828;                                                           % Costante gravitazionale Marte [km^3/s^2]
            mu_Phobos = 0.0007112;                                                      % Costante gravitazionale Phobos [km^3/s^2]
            mu_Deimos = 0.0000985;                                                      % Costante gravitazionale Deimos [km^3/s^2]
            mu_Sole = 1.3271e11;                                                        % Costante gravitazionale del Sole [km^3/s^2]

            mu = mu_Phobos/(mu_Marte+mu_Phobos);                                        % Costante gravitazionale del sistema sinodico Marte-Phobos
            % Adimensionale [/]

            distanza = 9378;                                                            % Distanza Marte-Phobos ---> Distance unit [Km]

            Raggio_Marte = 3389;                                                        % [Km]
            Raggio_Phobos = 11.267;                                                     % [Km]
            raggio_TS = 1.4959e8;                                                       % Terra-Sole [Km]
            raggio_MS = 2.2793e8;                                                       % Marte-Sole [Km]
            distanza_TM = 78e6;                                                         % Distanza Terra-Marte [Km]
            h = 400;                                                                    % Quota orbita iniziale [Km]

            altitudine =  Raggio_Marte + h;                                             % Quota orbita iniziale [Km]

            V_mod = sqrt((mu_Marte+mu_Phobos)/altitudine);                              % Velocità circolare orbita iniziale [Km/s]

            Raggio_influenza_Marte = raggio_MS * (massa_Marte/massa_Sole)^(2/5);        % Calcolo la sfera di influenza di Marte
            % perché voglio che r > Raggio_influenza_Marte [Km]

            vel_unit = sqrt((mu_Marte+mu_Phobos)/distanza);                             % Velocità tangenziale del primario nel sistema
            % sinodico --> Velocity unit [Km/s]

            omega = sqrt((mu_Marte + mu_Phobos) / distanza^3);                          % Velocita angolare del primario [rad/sec]

            ts = (2*pi)/omega;                                                          % Tempo dimensionale [sec]
            tau = ts*omega;                                                             % Tempo adimensionalizzato [/]
            tspan = linspace(0,tau,n_iter)/2;                                           % Vettore che contiene i vari passi temporali in un periodo
            t3 = (2*pi)/data.omega_3;                                                   % Tempo dimensionale Phobos [sec]

            %% ESCAPE AD UN IMPULSO

            R = Raggio_Marte + h;                                                       % [Km]

            Vel_Circolare_Satellite = sqrt(mu_Marte/R);                                 % Velocità circolare S/C alla quota h [Km/s]

            a_GTO = (raggio_MS+raggio_TS)/2;                                            % Semiasse ellisse Hohmann [Km]
            Eg_GTO = -mu_Sole/(2*a_GTO);                                                % Energia Hohmann
            V1H = sqrt(2*(Eg_GTO + mu_Sole/raggio_MS));                                 % Velocità che devo avere una volta uscito dalla
            % sfera di influenza di Marte per tornare sulla Terra [Km/s]

            Vel_Circolare_Marte = sqrt(mu_Sole/raggio_MS);                              % [Km/s]

            Vel_Inf = abs(V1H - Vel_Circolare_Marte);                                   % Appena esco dalla sfera di influenza di Marte 'eredito'
            % la sua velocità di trascinamento attorno al Sole [Km/s]
            Energia_iperbole = (Vel_Inf^2)/2 - mu_Marte/Raggio_influenza_Marte;

            Vel_iperbole = sqrt(2*(Energia_iperbole + mu_Marte/R));                     % Velocità che devo avere per immettermi nell'iperbole
            % (inizialmente lo S/C possiede la Vel_Circolare_Satellite
            % e non la Vel_iperbole) [Km/s]

            Vel_escape = sqrt(2) * Vel_Circolare_Satellite;                             % [Km/s]


            DeltaV1 = Vel_iperbole - Vel_Circolare_Satellite;                           % [Km/s]

            %% VARIABILI DELLA TRAIETTORIA

            % Imposta i parametri dell'algoritmo genetico

            variabili = 4;                                                              % Le variabili sono in ordine: dv,theta,phi,tf
            popolazione_size = 1000;         % ( check deve essere > 20 )               % Numero individui
            generazioni = 1;                                                            % Numero generazioni

            % Imposta i limiti per i gli elementi della popolazione
            % 1 --> DV
            % 2 --> Theta (phi sempre 0 - modello planare)
            % 3 --> Tau (tempo adimensionale in radianti)
            % 4 --> Theta_Deimos

            limite1 = [0.1, 0.5];
            limite2 = [280 , 330];
            limite3 = [15,  60];
            limite4 = [30 ,   70];


            % Genera la popolazione iniziale tale che sia rispettosa dei limiti imposti
            % per ciascun elemento della popolazione

            popolazione = zeros(popolazione_size, variabili);

            popolazione(:,1) = rand(popolazione_size, 1) .*...                          % Genero una popolazione formata da una matrice 1000x4
                (limite1(2)-limite1(1)) + limite1(1);                                   % tale che ogni colonna potrà avere valori randomici
            popolazione(:,2) = rand(popolazione_size, 1) .*...                          % ma sempre in modo che tali valori causali cadano
                (limite2(2)-limite2(1)) + limite2(1);                                   % all'interno dell'intervallo ammesso da ciascuna
            popolazione(:,3) = rand(popolazione_size, 1) .*...                          % variabile
                (limite3(2)-limite3(1)) + limite3(1);
            popolazione(:,4) = rand(popolazione_size, 1) .*...
                (limite4(2)-limite4(1)) + limite4(1);
            %             popolazione(:,5) = rand(popolazione_size, 1) .*...
            %                 (limite5(2)-limite5(1)) + limite5(1);
            %             popolazione(:,6) = rand(popolazione_size, 1) .*...
            %                 (limite6(2)-limite6(1)) + limite6(1);


            % data.th0P = popolazione(1,5)*pi/180;
            % data.th0D = popolazione(1,5)*pi/180;


            % figure('renderer','painters','position',[0,150,700,600]);                   % In questo modo tengo sempre aperta la figura
            % hold on;                                                                    % che plotterà l'errore ad ogni generazione
            % grid minor;
            % xlabel('Generation i-th')
            % ylabel('Error i-th')

            %% LOOP PRINCIPALE

            err = Inf;
            err1 = Inf;
            err2 = Inf;
            err3 = Inf;
            err4 = Inf;
            err5 = Inf;
            indici_migliori=1;
            max_err = Inf;                                                              % Definisco un errore iniziale pari a infinito, esso verrà
            % aggiornato ad ogni iterazione

            flag_crossover = false;                                                     % Definisco un flag per avere traccia dei miglioramenti
            % dovuti a crossover durante le varie generazioni

            flag_somma = false;                                                         % Definisco un flag per avere traccia dei miglioramenti
            % dovuti a somma algebrica tra gli elleli dei genitori
            % durante le varie generazioni


            i=0;

            while ~( err1(indici_migliori(1)) <= 0.1 &&  err2(indici_migliori(1)) <= 0.1 &&  err4(indici_migliori(1)) <= 0.3 &&  err5(indici_migliori(1)) <= 0.3)

                err = Inf;
                max_err = Inf;
                i=i+1;

                c_ini = pop2cini(popolazione,mu,altitudine,V_mod,distanza,vel_unit);
                yy = [c_ini(i,1); c_ini(i,2); c_ini(i,4); c_ini(i,5)];                  % questo vettore verrà passato alla funzione PBR4BP che analizza un caso planare con z=0 e vz=0

                %  data.th0P = popolazione(i,5)*pi/180;
                %    data.th0D = popolazione(randi(popolazione_size,1),4)*pi/180;

                options = odeset('RelTol',1e-6,'AbsTol',1e-6);

                for j = 1:popolazione_size

                    data.th0D = popolazione(randi(popolazione_size,1),4)*pi/180;

                    [tout, yout] = ode113(@(t,yy) PBR4BP(t,yy,data),...                     % Ad ogni iterazione entro con l'individuo della popolazione
                        linspace(0,popolazione(j,4),n_iter), c_ini(j,[1,2,4,5]), options);  % e propago, per un tempo pari alla quarta componente dell'individuo, l'orbita con ode113

                    r_target = [Raggio_influenza_Marte/distanza, 0];                % Raggio target: partendo da altitudine/dist voglio
                    % arrivare ai confini della sfera di influenza di Marte

                    yout_inerz = Syn2ECI(tout',yout,mu);
                    x_in = yout_inerz(:,1);
                    y_in = yout_inerz(:,2);


                    rf_syn  = yout(end,1:2);                                        % Raggio finale sinodico: set contiene le tre posizioni alla fine
                    % dell'integrazione con ode113 quindi è dove effettivamente
                    % siamo arrivati fissate le condizioni iniziali

                    vf_syn  = yout(end,3:4);                                        % Velocita finale sinodico: set contiene le velocità  di arrivo
                    % nelle tre direzioni

                    rf_inerz = yout_inerz(end,1:2);                                 % Raggio finale inerziale


                    vf_inerz = yout_inerz(end,3:4);                                 % Velocità finale inerziale

                    for t = 1:length(tout)                                          %Passo di temporale (sarebbero i radianti percorsi)

                        xP_in = cos(tout(t) + data.th0P);
                        yP_in = sin(tout(t) + data.th0P);

                        xD_in = data.DU_13_adim * cos(data.omega_3/data.omega_2*tout(t) + data.th0D);
                        yD_in = data.DU_13_adim * sin(data.omega_3/data.omega_2*tout(t) + data.th0D);

                        rPh_in(t,:) = [xP_in, yP_in];
                        rSC_in(t,:) = [x_in(t),y_in(t)];
                        rDe_in(t,:) = [xD_in, yD_in];

                        rSCPh_in(t) = norm(rSC_in(t,1:2) - rPh_in(t,1:2));
                        rSCDe_in(t) = norm(rSC_in(t,1:2) - rDe_in(t,1:2));

                    end

                    [mPh_in,indPh_in] = min(rSCPh_in);
                    [mDe_in,indDe_in] = min(rSCDe_in);

                    err1(j) = abs(norm(rf_inerz)  - norm(r_target));                  % Errore sul raggio finale

                    err2(j) = abs( norm(vf_inerz) * vel_unit - Vel_Inf );             % Errore sulla velocità finale

                    err3(j) = abs(atan2(vf_inerz(2), vf_inerz(1)));                   % Calcola l'angolo tra il vettore velocità e l'asse x in radianti

                    err4(j) = abs(mPh_in - (data.Raggio_Phobos/distanza)*1.1);        % Arrivo il più vicino possibile a Phobos cioè faccio un flyby a 0.05*distanza=469km

                    err5(j) = abs(mDe_in - (data.Raggio_Deimos/distanza)*1.1);

                    err6(j) = popolazione(j,1);                                       % Per minimizzare il DV

                    err(j) = err1(j) + err2(j) + err6(j) + err4(j) + err5(j);

                end

                if min(err) < max_err                                                   % Dopo la propagazione aggiorno la variabile max_err
                    max_err = min(err);                                                 % con il valore del minimo errore all'interno della
                    % singola generazione di individui

                    if flag_somma                                                       % Se il flag_somma è diventato vero allora c'è stato un
                        fprintf('miglioramento post somma: \n')                         % miglioramento post somma algebrica tra gli alleli dei
                    end                                                                 % genitori e lo stampo

                    if flag_crossover                                                   % Se il flag_crossover è diventato vero allora c'è stato
                        fprintf('miglioramento post crossover: \n')                     % un miglioramento post crossover dei genitori e lo stampo
                    end

                end

                [valori_migliori, indici_migliori] = sort(err);                         % Ordino gli errore dal più piccolo al più grande
                % e mi salvo gli indici relativi agli individui
                % migliori (errore più piccolo)
                [valori_migliori1, indici_migliori1] = sort(err1);
                [valori_migliori2, indici_migliori2] = sort(err2);
                [valori_migliori3, indici_migliori3] = sort(err3);
                [valori_migliori4, indici_migliori4] = sort(err4);
                [valori_migliori5, indici_migliori5] = sort(err5);

                n_elite = popolazione_size/4;                                           % Numero genitori elitari

                indici_migliori  = indici_migliori(1:n_elite);                          % Mi salvo soltanto gli indici dei primi n_elite migliori

                popolazione_ordinata = popolazione(indici_migliori,:);                  % Ordino la popolazione dall'individuo migliore al peggiore
                c_ini_save = c_ini(indici_migliori,:);
                popolazione_unica = popolazione_ordinata(1,:);                          % Seleziono il primo degli n_elite individui unici
                w = 2;
                while size(popolazione_unica,1) < n_elite
                    if w > size(popolazione_ordinata,1)                                 % se w > 250 allora c'è stato qualche individuo ripetuto
                        %             warning("La popolazione non contiene abbastanza individui unici." + ...
                        %               "La dimensione dell'élite è stata ridotta a %d.", size(popolazione_unica,1));
                        break;
                    end
                    if ~isequal(popolazione_ordinata(w,:),popolazione_unica(end,:))         % Se gli individui sono diversi entra nell'if
                        popolazione_unica = [popolazione_unica; popolazione_ordinata(w,:)]; % Aggiungo l'individuo alla popolazione unica
                    end
                    w = w+1;
                end

                if size(popolazione_unica,1) < n_elite                                                 % se la popolazione non contiene abbastanza individui unici

                    popolazione_ordinata_rimasta = popolazione_ordinata(w:end,:);                      % estraggo la parte rimanente della popolazione ordinata
                    popolazione_ordinata_rimasta_unica = unique(popolazione_ordinata_rimasta,'rows');  % elimino gli individui duplicati dalla popolazione rimanente
                    n_mancanti = n_elite - size(popolazione_unica,1);                                  % calcolo quanti individui mancano per arrivare a n_elite
                    n_da_aggiungere = min(n_mancanti, size(popolazione_ordinata_rimasta_unica,1));     % calcolo quanti individui posso aggiungere tra quelli rimanenti
                    popolazione_unica = [popolazione_unica;popolazione_ordinata_rimasta_unica(1:n_da_aggiungere,:)];                      % aggiungo gli individui alla popolazione unica

                end

                if min(err) <= max_err

                    % elite = popolazione_unica(1:n_elite,:);

                    elite   =  popolazione(indici_migliori(1:n_elite),:);                             % Creo gli elite che saranno i primi 'n_elite' migliori
                    % tra tutta la popolazione della singola generazione

                end

                n_nuovi = popolazione_size/4;                                               % Numero di elementi che vengono mutati per forza random

                results = table (i, elite(1,1), elite(1,2), elite(1,3),data.th0P*180/pi, elite(1,4),...
                    err(indici_migliori(1)),err1(indici_migliori(1)),...
                    err2(indici_migliori(1)),err4(indici_migliori(1)),err5(indici_migliori(1)),...
                    'VariableNames',{'i' 'DV' 'theta S/C' 'tf' 'thetaPh' 'thetaDe' 'err' 'err1' 'err2' 'err4' 'err5'}) %'err3'

                %     fprintf('% 4d % .4f % .4f % .4f % .4f % .4e % .4e % .4e  \n',...
                %     i,elite(1,:),min(err(indici_migliori(1))),min(err1(indici_migliori(1))),...
                %     min(err2(indici_migliori(1))),min(err3(indici_migliori(1))))

                % Ad ogni generazione stampo l'individuo migliore

                %     plot(i,min(err),'k.')                                                   % Plotto il minimo dell'errore che si ha ad ogni generazione
                %     drawnow
                %     pause(1e-3)


                %% Generazione Dadi 1,2,3 e 4,5,6

                % Dado_1 --> Selezione indice allele tra 1 e length(gene)
                % Dado_2 --> Selezione genitore 1
                % Dado_3 --> Selezione genitore 2

                % Dado_4 --> Determinazione probabilità di mutazione
                % Dado_5 --> Determinazione probabilità di mutazione
                % Dado_6 --> Selezione allele mutato

                popolazione = zeros(popolazione_size,variabili);                        % La mia nuova popolazione di 1000 individui
                % sarà formata da:

                popolazione(1:n_elite,:) = elite;                                       % 1) Le prime 250 posizioni saranno gli elite

                rows_to_keep = any(popolazione, 2);
                popolazione = popolazione(rows_to_keep, :);

                % Genero una popolazione randomica da 251 a n_iter

                popolazione(n_elite+1:2*n_elite,1)  = rand(n_elite, 1) .* (limite1(2)-limite1(1)) + limite1(1);

                popolazione(n_elite+1:2*n_elite,2)  = rand(n_elite, 1) .* (limite2(2)-limite2(1)) + limite2(1);

                popolazione(n_elite+1:2*n_elite,3)  = rand(n_elite, 1) .* (limite3(2)-limite3(1)) + limite3(1);

                popolazione(n_elite+1:2*n_elite,4)  = rand(n_elite, 1) .* (limite4(2)-limite4(1)) + limite4(1);

                % Faccio figliare non solo i 250 elitari ma anche i 250 nuovi random
                % (perché ho notato che facendo figliare solo gli elite la popolazione
                % non era ben distribuita ma tutti gli individui erano simili già dopo
                % le prime generazioni)

                for k=2*n_elite+1:popolazione_size                                      % k che va da 501 a 1000

                    ind1 = randi(variabili,1);                                          % Genero un numero a caso tra 1 e 4 che rappresenta
                    % l'allele che verrà cambiato

                    ind2 = randi([1 size(elite,1)],1,1);                                % Genero un numero a caso tra 1 e n_iter per scegliere
                    % uno dei n_iter genitori come padre

                    ind3 = randi([1 size(elite,1)],1,1);                                % Genero un numero a caso tra 1 e n_iter per scegliere
                    % uno dei n_iter genitori come madre

                    ind4 = rand(1,1);                                                   % Genero un numero decimale casuale tra 0 e 1


                    if ind2==ind3                                                       % Se i genitori sono uguali entra nel ciclo

                        while ind2==ind3                                                % Finché i genitori sono uguali cambia randomicamente
                            % la madre in modo da avere un (dv,theta,phi,tf)
                            ind3 = randi([1 size(elite,1)],1,1);                        % diverso dal padre

                        end

                    end

                    flag_somma = false;
                    flag_crossover=false;

                    if ind4 < 0.3                                                        % Si avrà il 30% di possibilità che i figli siano generati
                        flag_somma = true;
                        flag_crossover = false;                                          % come somma algebrica dei (dv,theta,phi,tf) dei genitori
                        % questo aiuta a restringere sempre di più il campo di
                        popolazione(k,1:4) = 0.5.* [ popolazione(ind2,1) +...            % ricerca della soluzione ottimale come se fosse un
                            popolazione(ind3,1), popolazione(ind2,2) +...                % metodo di bisezione. Se questo succede setto il
                            popolazione(ind3,2), popolazione(ind2,3) + ...               % flag_somma a 'true'
                            popolazione(ind3,3), popolazione(ind2,4) +...
                            popolazione(ind3,4)];

                    else

                        flag_crossover = true;
                        flag_somma = false;

                        popolazione(k,1:ind1) =  popolazione(ind2,1:ind1);                  % Altrimenti si avrà il 70% di possibilità che il figlio
                        popolazione(k,ind1+1:end) =  popolazione(ind3,ind1+1:end);          % erediterà un gene casuale dal padre e il resto dei geni
                        % dalla madre (che saranno entrambi elitari) o viceversa
                    end                                                                     % --> crossover <--

                end

                % Appena usciamo da questo for avremo una popolazione di 1000/1000
                % individui di cui 250 saranno elitari, 250 saranno random e n_iter
                % saranno generati con una probabilità del 70% tramite crossover dei
                % genitori e con una probabilità del 30% tramite media algebrica degli
                % elleli dei genitori

                % Mutazione della popolazione tranne gli elite

                for z=(n_elite+1):popolazione_size                                          % Mutazione

                    ind5 = rand(1,1);                                                       % Genero un numero decimale casuale tra 0 e 1

                    if ind5 <= 0.5                                                          % Si avrà il 50% di possibilità che tutti gli individui
                        % (tranne gli elite) possano mutare

                        ind6 = randi(variabili,1);                                          % Numero interi casuali da 1 a 4

                        % A seconda il valore di ind6 si sceglie quale allele cambierà:

                        switch ind6
                            case 1
                                popolazione(z,ind6) = rand(1,1) .* (limite1(2)-limite1(1)) + limite1(1);
                            case 2
                                popolazione(z,ind6) = rand(1,1) .* (limite2(2)-limite2(1)) + limite2(1);
                            case 3
                                popolazione(z,ind6) = rand(1,1) .* (limite3(2)-limite3(1)) + limite3(1);
                            case 4
                                popolazione(z,ind6) = rand(1,1) .* (limite4(2)-limite4(1)) + limite4(1);
                        end
                    end

                end


            end


            DeltaV = elite(1,1);
            DV_backup(b) = elite(1,1);
            MU12_backup(b) = data.mu;
            M3_backup(b) = data.m3_adim;
            RHO13_backup(b) = data.DU_13_adim;

        end
    end

end

%%  CREAZIONE HEATMAP

MU12_backup = [1.6606e-8,1.6606e-7,1.6606e-6,1.6606e-5,1.6606e-4,1.6606e-3,1.6606e-2,1.6606e-1];
M3_backup = [3.1299e-9,3.1299e-8,3.1299e-7,3.1299e-6,3.1299e-5,3.1299e-4,3.1299e-3,3.1299e-2,3.1299e-1];

% %%%%% D=2.5 %%%%%%
% dati=[
% 1.1401,1.0828,1.1777,1.1571,1.1383,1.1334,0.4648,0.1155;
% 1.1138,1.0933,1.1704,1.1536,1.1276,1.1027,0.4676,0.2006;
% 1.2334,1.1616,1.1542,1.2051,1.1647,1.1914,0.7332,0.3360;
% 1.1905,1.1909,1.2084,1.2011,1.2287,1.1996,0.6263,0.2402;
% 1.2065,1.2167,1.2125,1.2124,1.1925,1.1618,0.8184,0.2565;
% 1.1822,1.1607,1.1792,1.2406,1.2058,1.1995,0.8532,0.3595;
% 1.1722,1.2097,1.1808,1.1781,1.2235,1.1612,0.6079,0.4068;
% 1.1665,1.1836,1.1717,1.1939,1.2177,1.1718,0.7316,0.2953;
% 1.1610,1.1997,1.1649,1.1616,1.2497,1.1722,0.6228,0.5074];
%
% %%%%% D=2.5 %%%%%%
%
%
% %%%%%%%%%% D=2.25 %%%%%%%%%%%
% dati=[
% 1.0837,1.0964,1.0869,1.1097,0.9371,1.1186,0.6066,0.1013;
% 1.0917,1.0547,1.0597,1.0371,1.0518,1.0817,0.4189,0.1230;
% 1.1683,1.1452,1.2059,1.1433,1.1719,1.1721,0.5601,0.3040;
% 1.1590,1.1832,1.2469,1.1998,1.1569,1.1845,0.3504,0.1107;
% 1.1684,1.1808,1.1922,1.2085,1.2136,1.2125,0.3351,0.1706;
% 1.1632,1.1802,1.1793,1.1797,1.1736,1.1997,0.5466,0.4233;
% 1.2091,1.1676,1.1614,1.1946,1.1996,1.0565,0.4868,0.2739;
% 1.1833,1.1624,1.1883,1.1903,1.1591,1.1695,0.5510,0.3759;
% 1.1713,1.1775,1.1619,1.1577,1.1852,0.9847,0.5959,0.3198];
%
% %%%%%%%%%% D=2.25 %%%%%%%%%%%
%
%
%
%
% %%%%% D=2 %%%%%%%%%%
%
% dati = [
% 1.0873,1.0937,1.0759,0.9999,0.9311,0.9899,0.4086,0.1018;
% 1.0519,1.0656,1.0268,0.9134,0.7181,0.7869,0.6874,0.1392;
% 1.1811,1.1837,1.1978,1.1593,0.9697,1.1146,0.4522,0.2164;
% 1.1934,1.1799,1.2016,1.2009,1.1605,1.1707,0.5927,0.2491;
% 1.2032,1.1923,1.2034,1.2018,1.1989,1.1636,0.5392,0.1674;
% 1.2038,1.1984,1.1900,1.2016,1.2001,1.2078,0.8002,0.1936;
% 1.2018,1.2036,1.2028,1.2003,1.1913,1.1709,0.4654,0.1392;
% 1.2036,1.1986,1.2043,1.2037,1.2017,1.1958,0.7579,0.1602;
% 1.2118,1.2036,1.2026,1.2063,1.2095,1.1439,0.5829,0.1407];
%
%
% %%%% D=2 %%%%%%%%%%%
%
%
%
%
%
% %%%% D=1.75 %%%%%%%%%
%
% dati= [
%     0.4993,0.3808,0.3981,0.4203,0.4175,0.4599,0.4881,0.10965;
%     1.177,1.197,1.182,1.172,1.152,1.147,0.7782,0.1819;
%     1.171,1.179,1.189,1.183,1.172,1.066,0.9874,0.1855;
%     1.202,1.201,1.189,1.194,1.167,1.042,1.045,0.2047;
%     1.206,1.198,1.182,1.181,1.201,1.171,0.9753,0.157;
%     1.203,1.204,1.202,1.182,1.165,1.146,0.9727,0.1738;
%     1.202,1.178,1.185,1.191,1.188,0.87438,1.0351,0.2043;
%     1.191,1.204,1.202,1.203,1.194,1.111,1.049,0.223;
%     1.21,1.202,1.207,1.203,1.198,1.134,1.064,0.2094];
%
% %%%% D=1.75 %%%%%%%%%
%
%
%
% %%% D=1.5 %%%%%%%%%
%
%         dati = [
%         0.8344,0.6993,0.3192,0.4184,0.3807,0.4702,0.3095,0.1690;
%         1.1716,1.2148,1.192,1.2083,1.1887,1.182,0.9311,0.357;
%         1.1630,1.1655,1.222,1.2292,1.177,1.1927,1.005,0.4065;
%         1.204,1.181,1.2109,1.2011,1.1645,1.1956,1.1085,0.4514;
%         1.204,1.1796,1.194,1.187,1.2007,1.1803,1.019,0.5174;
%         1.206,1.2371,1.1999,1.1670,1.1859,1.2011,1.0469,0.5134;
%         1.202,1.2123,1.1849,1.162,1.182,1.201,1.089,0.4057;
%         1.209,1.1976,1.190,1.180,1.1862,1.1634,0.9817,0.41342;
%         1.216,1.212,1.205,1.1998,1.1991,1.1396,1.1313,0.5341];
%
% %%% D=1.5 %%%%%%%%%


%%%%% D=1.25 %%%%%%

% dati=[
% 0.8515,0.4868,0.4631,0.3030,0.5752,0.3602,0.2351,0.1201;
% 1.2133,1.0992,1.1897,1.1976,1.2951,0.5708,0.2412,0.2063;
% 1.1638,1.1471,1.1454,1.1421,1.2614,1.1120,0.1199,0.1485;
% 1.1804,1.1458,1.1763,1.1110,1.2296,1.1898,0.3001,0.1490;
% 1.2318,1.1318,1.2187,1.1820,1.2112,1.1881,0.2811,0.1995;
% 1.1990,1.2184,1.2609,1.2144,1.1854,1.2520,0.5337,0.2101;
% 1.2490,1.1340,1.1178,1.2485,1.2057,1.1666,0.5370,0.1881;
% 1.2037,1.1853,1.2226,1.1756,1.2146,1.1858,0.4761,0.2490;
% 1.2437,1.1886,1.1808,1.1978,1.1758,1.1941,0.7284,0.2463];


%%%%% D=1.25 %%%%%%


% %     MU12_backup = rand(10,1);
% %     M3_backup = rand(10,1);
% %     DV_backup = rand(10,10);
% %     RHO13_backup = [1,1,1.5,1.5,3];
%
%   Creazione di una griglia X-Y
[X, Y] = meshgrid(unique(MU12_backup), unique(M3_backup));

%  Creazione dei dati numerici per ogni coppia X-Y
%    dati = DV_backup;
%     dati = [dati(65:72);dati(57:64);dati(49:56);
%         dati(41:48);dati(33:40);dati(25:32);
%         dati(17:24);dati(9:16);dati(1:8)];

%  Creazione della heatmap
h = heatmap(unique(MU12_backup),flip(unique(M3_backup)),dati);

colormap('jet'); % Puoi sostituire 'jet' con la colormap che preferisci
colorbar; % Aggiungi una barra dei colori per visualizzare la scala dei colori

%   Aggiungi etichette agli assi, titolo e qualsiasi altra personalizzazione necessaria
xlabel('\mu_{12}');
ylabel('m_3');
title('\DeltaV');


%% CONDIZIONI INIZIALI ADIMENSIONALIZZATE

xf  =  elite(1,:);                                                           % xf conterrà (dv,theta,phi,tf) dell'individuo migliore

cond_ini = pop2cini(xf,mu,altitudine,V_mod,distanza,vel_unit);
yy = [cond_ini(1,1); cond_ini(1,2); cond_ini(1,4); cond_ini(1,5)];           % Caso planare per cui z=0 e vz=0 (PBR4BP)

%% PROPAGAZIONE CONDIZIONI INIZIALI OTTIMIZZATE CON ODE 113

n_iter = 2000;
[tout, yout] = ode113(@(t,yy) PBR4BP(t,yy,data),...                          % Propago l'orbita dell'individuo migliore (cioè
    linspace(0,xf(end,4),n_iter),cond_ini(1,[1,2,4,5]),options);             % quello che ha ottenuto un errore più piccolo e quindi
% è arrivato più vicino al target) con ode113

%% SALVATAGGIO VARIABILI NEL SISTEMA INERZIALE E SINODICO

cond_iniziali = yout(1,:);
cond_finali = yout(end,:);

Theta=linspace(0,xf(end,4),n_iter);                                         % Da 0 alla distanza angolare in radianti percorsa dallo S/C
Theta360 = linspace(0,2*pi,n_iter);                                         % Per plottare le orbite a 360° uso questo

yout_inerz = Syn2ECI(tout',yout,mu);                                        % Da sinodico a ECI (Earth Centered Inertial)

% vettore di stato nel sistema sinodico planare

x_syn = yout(:,1);
y_syn = yout(:,2);
z_syn = zeros(n_iter,1);
%z_syn = yout(:,3);
vx_syn = yout(:,3);
vy_syn = yout(:,4);
vz_syn = zeros(n_iter,1);
%vz_syn = yout(:,6);

% vettore di stato nel sistema inerziale planare

x_in = yout_inerz(:,1);
y_in = yout_inerz(:,2);
z_in = zeros(n_iter,1);
%z_in = yout_inerz(:,3);
vx_in = yout_inerz(:,3);
vy_in = yout_inerz(:,4);
vz_in = zeros(n_iter,1);
%vz_in = yout_inerz(:,6);

% Definisci la posizione e il raggio del cerchio che rappresenterà Marte
xCenter = -mu;
yCenter = 0;
raggio = Raggio_Marte/distanza;

%% PLOT 4 FOTOGRAFIE INERZIALE

figure('Renderer','painters','position',[100 100 1200 600])
hold on
grid minor
axis equal

plot3(-mu,0,0,'r.','MarkerSize',50)                                               % Plotto Marte in posizione -mu nel sistema sinodico
plot3(1-mu,0,0,'k.','MarkerSize',20)                                              % Plotto Phobos in posizione 1-mu nel sistema sinodico

plot3(-mu*cos(Theta360),-mu*sin(Theta360),zeros(1,length(Theta360)),'r--')        % Plotto l'orbita di Marte
plot3((1-mu)*cos(Theta360),(1-mu)*sin(Theta360),zeros(1,length(Theta360)),'k--')  % Plotto l'orbita di Phobos
plot3(data.DU_13_adim*cos(Theta360),data.DU_13_adim*sin(Theta360),zeros(1,length(Theta360)),'g--')  % Plotto l'orbita di Deimos
plot3((altitudine/distanza)*cos(Theta360),(altitudine/distanza)*sin(Theta360), ...
    zeros(1,length(Theta360)),'b--')                                              % Plotto l'orbita iniziale dello S/C


% Inizializzo le coordinate del pallino che rappresenta Phobos e Deimos

theta_iniziale_P = data.th0P;                                                     % inizio la traiettoria quando Phobos si trova a 290°
theta_iniziale_D = data.th0D;                                                     % 32° per il combinato

xP_in = cos(theta_iniziale_P);                                                    % Posiziona il pallino all'inizio dell'orbita1
yP_in = sin(theta_iniziale_P);

xD_in = data.DU_13_adim * cos(theta_iniziale_D-theta_iniziale_P);                 % Posiziona il pallino all'inizio dell'orbita
yD_in = data.DU_13_adim * sin(theta_iniziale_D-theta_iniziale_P);

% Disegno il pallino iniziale
pallinoP = plot(xP_in, yP_in, 'k.', 'MarkerSize', 20);
pallinoD = plot(xD_in, yD_in, 'g.', 'MarkerSize', 20);

for t = 1:length(tout)                                                             %Passo di temporale (sarebbero i radianti percorsi)

    % Calcolo le nuove coordinate del pallino
    % theta = theta_iniziale + theta;
    xP_in = cos(tout(t) + theta_iniziale_P);
    yP_in = sin(tout(t) + theta_iniziale_P);

    xD_in = data.DU_13_adim * cos(data.omega_3/data.omega_2*tout(t) + (theta_iniziale_D-theta_iniziale_P) );
    yD_in = data.DU_13_adim * sin(data.omega_3/data.omega_2*tout(t) + (theta_iniziale_D-theta_iniziale_P) );

    rPh_in(t,:) = [xP_in, yP_in];
    rDe_in(t,:) = [xD_in, yD_in];
    rSC_in(t,:) = [x_in(t),y_in(t)];

    rSCPh_in(t) = norm(rSC_in(t,1:2) - rPh_in(t,1:2));
    rSCDe_in(t) = norm(rSC_in(t,1:2) - rDe_in(t,1:2));

end

[mPh_in,indPh_in] = min(rSCPh_in);
[mDe_in,indDe_in] = min(rSCDe_in);

subplot(2,2,1) %Posizione iniziale
hold on
grid minor
axis equal
title('Starting position')
xlabel(char(958)); %csi
%ylabel(char(951)); %eta
ylabel(char(951), 'Rotation', 360);

plot3(-mu,0,0,'r.','MarkerSize',50)
plot3(-mu*cos(Theta360),-mu*sin(Theta360),zeros(1,length(Theta360)),'r--')                          % Plotto l'orbita di Marte
plot3((1-mu)*cos(Theta360),(1-mu)*sin(Theta360),zeros(1,length(Theta360)),'k--')                    % Plotto l'orbita di Phobos
plot3(data.DU_13_adim*cos(Theta360),data.DU_13_adim*sin(Theta360),zeros(1,length(Theta360)),'g--')  % Plotto l'orbita di Deimos
plot3((altitudine/distanza)*cos(Theta360),(altitudine/distanza)*sin(Theta360), ...                  % Plotto l'orbita iniziale
    zeros(1,length(Theta360)),'b--')

% Disegna il cerchio di Marte
rectangle('Position', [xCenter - raggio, yCenter - raggio, 2*raggio, 2*raggio],...
    'Curvature', [1, 1], 'FaceColor', 'r')
plot3(rSC_in(1,1),rSC_in(1,2),0,'b.','MarkerSize',10);
plot3(rPh_in(1,1),rPh_in(1,2),0,'k.','MarkerSize',10);
plot3(rDe_in(1,1),rDe_in(1,2),0,'g.','MarkerSize',10);

set(gca, 'fontsize', 14)

subplot(2,2,2) % Minima distanza Phobos
hold on
grid minor
axis equal
title('Minimum distance from Phobos')
xlabel(char(958));
ylabel(char(951), 'Rotation', 360);

plot3(-mu,0,0,'r.','MarkerSize',50)
plot3(-mu*cos(Theta360),-mu*sin(Theta360),zeros(1,length(Theta360)),'r--')                          % Plotto l'orbita di Marte
plot3((1-mu)*cos(Theta360),(1-mu)*sin(Theta360),zeros(1,length(Theta360)),'k--')                    % Plotto l'orbita di Phobos
plot3(data.DU_13_adim*cos(Theta360),data.DU_13_adim*sin(Theta360),zeros(1,length(Theta360)),'g--')  % Plotto l'orbita di Deimos
plot3((altitudine/distanza)*cos(Theta360),(altitudine/distanza)*sin(Theta360), ...                  % Plotto l'orbita iniziale
    zeros(1,length(Theta360)),'b--')

% Disegna il cerchio di Marte
rectangle('Position', [xCenter - raggio, yCenter - raggio, 2*raggio, 2*raggio],...
    'Curvature', [1, 1], 'FaceColor', 'r')
plot3(rSC_in(1:indPh_in,1),rSC_in(1:indPh_in,2),zeros(indPh_in,1),'b--');
plot3(rSC_in(indPh_in,1),rSC_in(indPh_in,2),zeros(indPh_in,1),'b.','MarkerSize',10);
plot3(rPh_in(indPh_in,1),rPh_in(indPh_in,2),zeros(indPh_in,1),'k.','MarkerSize',10);
plot3(rDe_in(indPh_in,1),rDe_in(indPh_in,2),zeros(indPh_in,1),'g.','MarkerSize',10);

set(gca, 'fontsize', 14)

subplot(2,2,3) % Minima distanza Deimos
hold on
grid minor
axis equal
title('Minimum distance from Deimos')
xlabel(char(958));
ylabel(char(951), 'Rotation', 360);

plot3(-mu,0,0,'r.','MarkerSize',50)
plot3(-mu*cos(Theta360),-mu*sin(Theta360),zeros(1,length(Theta360)),'r--')                          % Plotto l'orbita di Marte
plot3((1-mu)*cos(Theta360),(1-mu)*sin(Theta360),zeros(1,length(Theta360)),'k--')                    % Plotto l'orbita di Phobos
plot3(data.DU_13_adim*cos(Theta360),data.DU_13_adim*sin(Theta360),zeros(1,length(Theta360)),'g--')  % Plotto l'orbita di Deimos
plot3((altitudine/distanza)*cos(Theta360),(altitudine/distanza)*sin(Theta360), ...                  % Plotto l'orbita iniziale
    zeros(1,length(Theta360)),'b--')

% Disegna il cerchio di Marte
rectangle('Position', [xCenter - raggio, yCenter - raggio, 2*raggio, 2*raggio],...
    'Curvature', [1, 1], 'FaceColor', 'r')
plot3(rSC_in(1:indDe_in,1),rSC_in(1:indDe_in,2),zeros(indDe_in,1),'b--');
plot3(rSC_in(indDe_in,1),rSC_in(indDe_in,2),zeros(indDe_in,1),'b.','MarkerSize',10);
plot3(rPh_in(indDe_in,1),rPh_in(indDe_in,2),zeros(indDe_in,1),'k.','MarkerSize',10);
plot3(rDe_in(indDe_in,1),rDe_in(indDe_in,2),zeros(indDe_in,1),'g.','MarkerSize',10);

set(gca, 'fontsize', 14)

subplot(2,2,4) %Posizione finale
hold on
grid minor
axis equal
title('Final position')
xlabel(char(958));
ylabel(char(951), 'Rotation', 360);

plot3(-mu,0,0,'r.','MarkerSize',50)
plot3(-mu*cos(Theta360),-mu*sin(Theta360),zeros(1,length(Theta360)),'r--')                          % Plotto l'orbita di Marte
plot3((1-mu)*cos(Theta360),(1-mu)*sin(Theta360),zeros(1,length(Theta360)),'k--')                    % Plotto l'orbita di Phobos
plot3(data.DU_13_adim*cos(Theta360),data.DU_13_adim*sin(Theta360),zeros(1,length(Theta360)),'g--')  % Plotto l'orbita di Deimos
plot3((altitudine/distanza)*cos(Theta360),(altitudine/distanza)*sin(Theta360), ...                  % Plotto l'orbita iniziale
    zeros(1,length(Theta360)),'b--')
plot3((Raggio_influenza_Marte/distanza)*cos(Theta360),(Raggio_influenza_Marte/distanza)*sin(Theta360),zeros(1,length(Theta360)),'k-.')    %Plotto SOI


% Disegna il cerchio di Marte
rectangle('Position', [xCenter - raggio, yCenter - raggio, 2*raggio, 2*raggio],...
    'Curvature', [1, 1], 'FaceColor', 'r')
plot3(rSC_in(1:end,1),rSC_in(1:end,2),zeros(length(rSC_in),1),'b--');
plot3(rSC_in(end,1),rSC_in(end,2),0,'b.','MarkerSize',10);
plot3(rPh_in(end,1),rPh_in(end,2),0,'k.','MarkerSize',10);
plot3(rDe_in(end,1),rDe_in(end,2),0,'g.','MarkerSize',10);

set(gca, 'fontsize', 14)

hold off

sgtitle('Inertial system')                                                            % Titolo ai subplot

%% PLOT DEL FLYBY DI PHOBOS INERZIALE

if anim == 1

    figure('Renderer','painters','position',[100 100 1200 600])
    hold on
    grid minor
    axis equal
    title('Inertial Phobos flyby')
    xlabel(char(958));
    ylabel(char(951), 'Rotation', 360);

    xlim([rPh_in(indPh_in,1)-.5 rPh_in(indPh_in,1)+.5]);
    ylim([rPh_in(indPh_in,2)-.5 rPh_in(indPh_in,2)+.5]);

    plot3((1-mu)*cos(Theta360),(1-mu)*sin(Theta360),zeros(1,length(Theta360)),'k--')
    plot3(rPh_in(indPh_in,1),rPh_in(indPh_in,2),zeros(indPh_in,1),'k.','MarkerSize',50);
    plot3(rSC_in(1:indPh_in,1),rSC_in(1:indPh_in,2),zeros(indPh_in,1),'b--');
    plot3(rSC_in(indPh_in,1),rSC_in(indPh_in,2),zeros(indPh_in,1),'b.','MarkerSize',20);

    hold off

else

    % Creazione del video
    video = VideoWriter('my_video.mp4', 'MPEG-4');
    video.FrameRate = 60;                                                % Imposta la frequenza di aggiornamento a tot fps
    video.Quality = 100;
    open(video);

    figure('Renderer','painters','position',[100 100 1200 600])
    hold on
    grid minor
    axis equal
    title('Inertial Phobos flyby')
    xlabel(char(958));
    ylabel(char(951), 'Rotation', 360);

    theta_iniziale_P = data.th0P;

    % Disegno il pallino iniziale
    pallinoP = plot(xP_in, yP_in, 'k.', 'MarkerSize', 50);

    plot3((1-mu)*cos(Theta360),(1-mu)*sin(Theta360),zeros(1,length(Theta360)),'k--')   % Plotto l'orbita di Phobos

    % Animazione del movimento
    for tPh = indPh_in - 300 : 1 : indPh_in + 300                          % Plotto i passi temporali immediatamente prima e dopo il flyby

        % Calcolo le nuove coordinate del pallino
        % theta = theta_iniziale + theta;
        xP_in = cos(tout(tPh) + theta_iniziale_P);
        yP_in = sin(tout(tPh) + theta_iniziale_P);

        hp_P_in(tPh,:) = [xP_in, yP_in];
        hp_SC_in(tPh,:) = [x_in(tPh),y_in(tPh)];

        % Aggiorno la posizione del pallino
        set(pallinoP, 'XData', xP_in, 'YData', yP_in);

        % Aggiorno i limiti degli assi in base alla posizione corrente del pallino
        xlim([xP_in-.5 xP_in+.5]);
        ylim([yP_in-.5 yP_in+.5]);

        %view(30,30)

        hp = plot3(x_in(1:tPh),y_in(1:tPh),z_in(1:tPh),'r-');

        % Aggiorno la figura
        pause(0.1)

        % Aggiungi un frame al video
        frame = getframe(gcf);
        writeVideo(video, frame);

    end

    hold off
    close(video);

end


%% PLOT DEL FLYBY DI DEIMOS INERZIALE

if anim == 1

    figure('Renderer','painters','position',[100 100 1200 600])
    hold on
    grid minor
    axis equal
    title('Inertial Deimos flyby')
    xlabel(char(958));
    ylabel(char(951), 'Rotation', 360);

    xlim([rDe_in(indDe_in,1)-.5 rDe_in(indDe_in,1)+.5]);
    ylim([rDe_in(indDe_in,2)-.5 rDe_in(indDe_in,2)+.5]);

    plot3(data.DU_13_adim*cos(Theta360),data.DU_13_adim*sin(Theta360),zeros(1,length(Theta360)),'g--')
    plot3(rDe_in(indDe_in,1),rDe_in(indDe_in,2),zeros(indDe_in,1),'g.','MarkerSize',50);
    plot3(rSC_in(1:indDe_in,1),rSC_in(1:indDe_in,2),zeros(indDe_in,1),'b--');
    plot3(rSC_in(indDe_in,1),rSC_in(indDe_in,2),zeros(indDe_in,1),'b.','MarkerSize',20);

    hold off

else

    % Creazione del video
    video = VideoWriter('my_video.mp4', 'MPEG-4');
    video.FrameRate = 60;                                                % Imposta la frequenza di aggiornamento a 30 fps
    video.Quality = 100;
    open(video);

    figure('Renderer','painters','position',[100 100 1200 600])
    hold on
    grid minor
    axis equal
    title('Inertial Deimos flyby')
    xlabel(char(958));
    ylabel(char(951), 'Rotation', 360);

    theta_iniziale_D = data.th0D;

    % Disegno il pallino iniziale
    pallinoD = plot(xD_in, yD_in, 'g.', 'MarkerSize', 50);

    plot3(data.DU_13_adim*cos(Theta360),data.DU_13_adim*sin(Theta360),zeros(1,length(Theta360)),'g--')                    % Plotto l'orbita di Phobos

    if indDe_in < 6, indDe_in = 6; end
    %if indDe_in > length(t)-4, indDe_in = length(t)-4; end

    % Animazione del movimento
    for tDe = indDe_in - 300 : 1 : indDe_in + 300                        % Plotto i passi temporali immediatamente prima e dopo il flyby

        % Calcolo le nuove coordinate del pallino
        % theta = theta_iniziale + theta;
        xD_in = data.DU_13_adim * cos(data.omega_3/data.omega_2*tout(tDe) + (theta_iniziale_D-theta_iniziale_P));
        yD_in = data.DU_13_adim * sin(data.omega_3/data.omega_2*tout(tDe) + (theta_iniziale_D-theta_iniziale_P));

        hp_D_in(tDe,:) = [xD_in, yD_in];
        hp_SC_in(tDe,:) = [x_in(tDe),y_in(tDe)];

        % Aggiorno la posizione del pallino
        set(pallinoD, 'XData', xD_in, 'YData', yD_in);

        % Aggiorno i limiti degli assi in base alla posizione corrente del pallino
        xlim([xD_in-.5 xD_in+.5]);
        ylim([yD_in-.5 yD_in+.5]);

        hp = plot3(x_in(1:tDe),y_in(1:tDe),z_in(1:tDe),'r-');

        % Aggiorno la figura
        pause(0.1)

        % Aggiungi un frame al video
        frame = getframe(gcf);
        writeVideo(video, frame);
    end
    hold off

    %Chiudi il video
    close(video);

end

% % subplot(2,2,1);
% % subplot(2,2,2);
% % subplot(2,2,3);
% % subplot(2,2,4);
% % exportgraphics(gcf, 'Inerziale_flyby_Phobos_Deimos_theta230.eps', 'Resolution', 300);

%% PLOT DELL'INTERA TRAIETTORIA INERZIALE

figure('Renderer','painters','position',[100 100 1200 600])
hold on
grid minor
axis equal
xlabel(char(958));
ylabel(char(951), 'Rotation', 360);

% Disegna il cerchio di Marte
rectangle('Position', [xCenter - raggio, yCenter - raggio, 2*raggio, 2*raggio],...
    'Curvature', [1, 1], 'FaceColor', 'r')
plot3(0,0,0,'r.','MarkerSize',50)                                               % Plotto Marte in posizione (0,0) nel sistema inerziale

% plot3(rPh_in(end,1),rPh_in(end,2),0,'k.','MarkerSize',10)
% plot3(rDe_in(end,1),rDe_in(end,2),0,'g.','MarkerSize',10)

plot3(rPh_in(indPh_in,1),rPh_in(indPh_in,2),0,'k.','MarkerSize',20)             % Plotto Phobos nella sua posizione al momento del flyby nel sistema inerziale
plot3(rDe_in(indDe_in,1),rDe_in(indDe_in,2),0,'g.','MarkerSize',20)             % Plotto Deimos nella sua posizione al momento del flyby nel sistema inerziale


%plot3(1-mu,0,0,'k.','MarkerSize',10)                                             % Plotto Phobos in posizione 1-mu nel sistema sinodico
%plot3(data.DU_13_adim,0,0,'g.','MarkerSize',10)                                   % Plotto Deimos in posizione DU_13_adim

plot3(-mu*cos(Theta360),-mu*sin(Theta360),zeros(1,length(Theta360)),'r--')        % Plotto l'orbita di Marte
plot3(cos(Theta360),sin(Theta360),zeros(1,length(Theta360)),'k--')                % Plotto l'orbita di Phobos
plot3(data.DU_13_adim*cos(Theta360),data.DU_13_adim*sin(Theta360),zeros(1,length(Theta360)),'g--')  % Plotto l'orbita di Deimos
plot3((altitudine/distanza)*cos(Theta360),(altitudine/distanza)*sin(Theta360), ...
    zeros(1,length(Theta360)),'b--')                                              % Plotto l'orbita iniziale dello S/C

if norm(r_target) > 30
    plot3((Raggio_influenza_Marte/distanza)*cos(Theta360), ...
        (Raggio_influenza_Marte/distanza)*sin(Theta360), zeros(1,length(Theta360)),'y.-')  % Plotto la sfera di influenza di Marte
end

plot3(0,0,0,'y.')                                                                  % Plotto il CM del sistema inerziale
%view(30,30)
view(0,90)
plot3(yout_inerz(1,1),yout_inerz(1,2),0,'b*')                                      % Punto di partenza S\C
plot3(yout_inerz(end,1),yout_inerz(end,2),0,'c*')                                  % Punto finale S\C

if anim == 1
    plot3(x_in,y_in,z_in)                                                          % Plotto la traiettoria
else
    comet3(x_in,y_in,z_in)                                                         % Plotto la traiettoria
end

title('Mars-Phobos-Deimos 2D Inertial System')
xlabel(char(958)); ylabel(char(951), 'Rotation', 360);
h = legend('Mars','Phobos','Deimos','Mars orbit','Phobos orbit','Deimos orbit', ...
    'Initial orbit S/C','SOI Mars','CM Inertial','Starting point','Final point','Trajectory'); %SOI Marte',

% Creazione della legenda
lgd = legend('Location', 'northwest');
numColumns = 2;  % Numero di colonne desiderate nella legenda
lgd.NumColumns = numColumns;

%set(gca, 'View', [-360 90]); % Ruota la visualizzazione
%     set(gca, 'fontsize', 14)
%     set(h,'fontsize',14)

hold off

%% PLOT VELOCITA INERZIALI

figure('Renderer','painters','position',[100 100 1200 600])
hold on
grid minor
plot(vx_in);
plot(vy_in);
%plot(vz_in);
v_in_mod = sqrt(vx_in.^2 + vy_in.^2);
plot(v_in_mod,'LineWidth',2);
xlabel('time steps')
ylabel('Velocity');
yline(Vel_Inf/vel_unit);
xline(indPh_in,'Color', 'green');
xline(indDe_in,'Color', 'green');
h=legend('VX\_in','VY\_in','V\_in\_mod','V\infty','Minimum from Phobos','Minimum from Deimos');
title('Inertial velocities')
set(gca, 'fontsize', 14)
set(h,'fontsize',14)
hold off

%% PLOT POSIZIONI INERZIALI

figure('Renderer','painters','position',[100 100 1200 600])
hold on
grid minor
plot(x_in);
plot(y_in);
%plot(vz_in);
r_in_mod = sqrt(x_in.^2 + y_in.^2);
plot(r_in_mod,'LineWidth',2);
xlabel('time steps')
ylabel(char(958), 'Rotation', 360); %csi
legend('X\_in','Y\_in','R\_in\_mod')
title('Inertial distances')
hold off


%% PLOT 4 FOTOGRAFIE SINODICO

figure('Renderer','painters','position',[100 100 1200 600])                     % 4 fotografie nel sistema sinodico
hold on
grid minor
axis equal

rectangle('Position', [xCenter - raggio, yCenter - raggio, 2*raggio, 2*raggio],...
    'Curvature', [1, 1], 'FaceColor', 'r')
plot3(-mu,0,0,'r.','MarkerSize',50)                                              % Plotto Marte in posizione -mu nel sistema sinodico
% plot3(1-mu,0,0,'k.','MarkerSize',20)                                           % Plotto Phobos in posizione 1-mu nel sistema sinodico
% plot3(data.DU_13_adim,0,0,'g.','MarkerSize',20)
plot3(-mu*cos(Theta360),-mu*sin(Theta360),zeros(1,length(Theta360)),'r--')           % Plotto l'orbita di Marte
plot3((1-mu)*cos(Theta360),(1-mu)*sin(Theta360),zeros(1,length(Theta360)),'k--')     % Plotto l'orbita di Phobos
plot3(data.DU_13_adim*cos(Theta360),data.DU_13_adim*sin(Theta360),zeros(1,length(Theta360)),'g--')  % Plotto l'orbita di Deimos

plot3((altitudine/distanza)*cos(Theta360),(altitudine/distanza)*sin(Theta360), ...
    zeros(1,length(Theta360)),'b--')                                             % Plotto l'orbita iniziale dello S/C


% Inizializzo le coordinate del pallino che rappresenta Phobos e Deimos

theta_iniziale_P = data.th0P;
theta_iniziale_D = data.th0D;

xP_syn = (1-mu) * cos(theta_iniziale_P);                                         % Posiziona il pallino all'inizio dell'orbita
yP_syn = (1-mu) * sin(theta_iniziale_P);

xD_syn = data.DU_13_adim * cos(theta_iniziale_D-theta_iniziale_P);               % Posiziona il pallino all'inizio dell'orbita
yD_syn = data.DU_13_adim * sin(theta_iniziale_D-theta_iniziale_P);

% Disegno il pallino iniziale
pallinoP = plot(xP_syn, yP_syn, 'k.', 'MarkerSize', 10);
pallinoD = plot(xD_syn, yD_syn, 'g.', 'MarkerSize', 10);

% Animazione del movimento
for t = 1:length(tout)                                                            % Passo di temporale (sarebbero i radianti percorsi)

    % Calcolo le nuove coordinate del pallino

    % Phobos si muove con velocità angolare unitaria nel sinodico
    xP_syn = (1-mu) * cos(theta_iniziale_P);
    yP_syn = (1-mu) * sin(theta_iniziale_P);

    % Deimos si muove con circa 1/4 della velocità angolare di Phobos per
    % cui, visto che il sistema sinodico si muove con velocità angolare
    % unitaria, Deimos sembrerà avere una velocità angolare negativa
    xD_syn = data.DU_13_adim * cos( (data.omega_3/data.omega_2 - 1) *tout(t) + (theta_iniziale_D-theta_iniziale_P));
    yD_syn = data.DU_13_adim * sin( (data.omega_3/data.omega_2 - 1) *tout(t) + (theta_iniziale_D-theta_iniziale_P));

    rPh_syn(t,:) = [xP_syn, yP_syn];
    rDe_syn(t,:) = [xD_syn, yD_syn];
    rSC_syn(t,:) = [x_syn(t),y_syn(t)];

    rSCPh_syn(t) = norm(rSC_syn(t,1:2) - rPh_syn(t,1:2));
    rSCDe_syn(t) = norm(rSC_syn(t,1:2) - rDe_syn(t,1:2));

end

[mPh_syn,indPh_syn] = min(rSCPh_syn);
[mDe_syn,indDe_syn] = min(rSCDe_syn);

subplot(2,2,1) %Posizione iniziale
hold on
grid minor
axis equal
title('Starting position')
xlabel(char(958));
ylabel(char(951), 'Rotation', 360);

plot3(-mu,0,0,'r.','MarkerSize',50)
plot3(-mu*cos(Theta360),-mu*sin(Theta360),zeros(1,length(Theta360)),'r--')                          % Plotto l'orbita di Marte
plot3((1-mu)*cos(Theta360),(1-mu)*sin(Theta360),zeros(1,length(Theta360)),'k--')                    % Plotto l'orbita di Phobos
plot3(data.DU_13_adim*cos(Theta360),data.DU_13_adim*sin(Theta360),zeros(1,length(Theta360)),'g--')  % Plotto l'orbita di Deimos
plot3((altitudine/distanza)*cos(Theta360),(altitudine/distanza)*sin(Theta360), ...                  % Plotto l'orbita iniziale
    zeros(1,length(Theta360)),'b--')

plot3(rSC_syn(1,1),rSC_syn(1,2),0,'b.','MarkerSize',10);
plot3(rPh_syn(1,1),rPh_syn(1,2),0,'k.','MarkerSize',10);
plot3(rDe_syn(1,1),rDe_syn(1,2),0,'g.','MarkerSize',10);

subplot(2,2,2) % Minima distanza Phobos
hold on
grid minor
axis equal
title('Minimum distance from Phobos')
xlabel(char(958));
ylabel(char(951), 'Rotation', 360);

plot3(-mu,0,0,'r.','MarkerSize',50)
plot3(-mu*cos(Theta360),-mu*sin(Theta360),zeros(1,length(Theta360)),'r--')                          % Plotto l'orbita di Marte
plot3((1-mu)*cos(Theta360),(1-mu)*sin(Theta360),zeros(1,length(Theta360)),'k--')                    % Plotto l'orbita di Phobos
plot3(data.DU_13_adim*cos(Theta360),data.DU_13_adim*sin(Theta360),zeros(1,length(Theta360)),'g--')  % Plotto l'orbita di Deimos
plot3((altitudine/distanza)*cos(Theta360),(altitudine/distanza)*sin(Theta360), ...                  % Plotto l'orbita iniziale
    zeros(1,length(Theta360)),'b--')

plot3(rSC_syn(1:indPh_syn,1),rSC_syn(1:indPh_syn,2),zeros(indPh_syn,1),'b--');
plot3(rSC_syn(indPh_syn,1),rSC_syn(indPh_syn,2),zeros(indPh_syn,1),'b.','MarkerSize',10);
plot3(rPh_syn(indPh_syn,1),rPh_syn(indPh_syn,2),zeros(indPh_syn,1),'k.','MarkerSize',10);
plot3(rDe_syn(indPh_syn,1),rDe_syn(indPh_syn,2),zeros(indPh_syn,1),'g.','MarkerSize',10);

subplot(2,2,3) % Minima distanza Deimos
hold on
grid minor
axis equal
title('Minimum distance from Deimos')
xlabel(char(958));
ylabel(char(951), 'Rotation', 360);

plot3(-mu,0,0,'r.','MarkerSize',50)
plot3(-mu*cos(Theta360),-mu*sin(Theta360),zeros(1,length(Theta360)),'r--')                          % Plotto l'orbita di Marte
plot3((1-mu)*cos(Theta360),(1-mu)*sin(Theta360),zeros(1,length(Theta360)),'k--')                    % Plotto l'orbita di Phobos
plot3(data.DU_13_adim*cos(Theta360),data.DU_13_adim*sin(Theta360),zeros(1,length(Theta360)),'g--')  % Plotto l'orbita di Deimos
plot3((altitudine/distanza)*cos(Theta360),(altitudine/distanza)*sin(Theta360), ...                  % Plotto l'orbita iniziale
    zeros(1,length(Theta360)),'b--')

plot3(rSC_syn(1:indDe_syn,1),rSC_syn(1:indDe_syn,2),zeros(indDe_syn,1),'b--');
plot3(rSC_syn(indDe_syn,1),rSC_syn(indDe_syn,2),zeros(indDe_syn,1),'b.','MarkerSize',10);
plot3(rPh_syn(indDe_syn,1),rPh_syn(indDe_syn,2),zeros(indDe_syn,1),'k.','MarkerSize',10);
plot3(rDe_syn(indDe_syn,1),rDe_syn(indDe_syn,2),zeros(indDe_syn,1),'g.','MarkerSize',10);

subplot(2,2,4) %Posizione finale
hold on
grid minor
axis equal
title('Final position')
xlabel(char(958));
ylabel(char(951), 'Rotation', 360);

plot3(-mu,0,0,'r.','MarkerSize',50)
plot3(-mu*cos(Theta360),-mu*sin(Theta360),zeros(1,length(Theta360)),'r--');                          % Plotto l'orbita di Marte
plot3((1-mu)*cos(Theta360),(1-mu)*sin(Theta360),zeros(1,length(Theta360)),'k--');                   % Plotto l'orbita di Phobos
plot3(data.DU_13_adim*cos(Theta360),data.DU_13_adim*sin(Theta360),zeros(1,length(Theta360)),'g--')  % Plotto l'orbita di Deimos
plot3((altitudine/distanza)*cos(Theta360),(altitudine/distanza)*sin(Theta360), ...                  % Plotto l'orbita iniziale
    zeros(1,length(Theta360)),'b--');
h = plot3((Raggio_influenza_Marte/distanza)*cos(Theta360),(Raggio_influenza_Marte/distanza)*sin(Theta360),zeros(1,length(Theta360)),'k-.')    %Plotto SOI

plot3(rSC_syn(1:end,1),rSC_syn(1:end,2),zeros(length(rSC_syn),1),'b--');
plot3(rSC_syn(end,1),rSC_syn(end,2),0,'b.','MarkerSize',10);
plot3(rPh_syn(end,1),rPh_syn(end,2),0,'k.','MarkerSize',10);
plot3(rDe_syn(end,1),rDe_syn(end,2),0,'g.','MarkerSize',10);

hold off

sgtitle('Synodic system')

%% FLYBY DI PHOBOS SINODICO

if anim == 1

    figure('Renderer','painters','position',[100 100 1200 600])
    hold on
    grid minor
    axis equal
    title('Synodic Phobos flyby')
    xlabel(char(958));
    ylabel(char(951), 'Rotation', 360);

    xlim([rPh_syn(indPh_syn,1)-.5 rPh_syn(indPh_syn,1)+.5]);
    ylim([rPh_syn(indPh_syn,2)-.5 rPh_syn(indPh_syn,2)+.5]);

    plot3((1-mu)*cos(Theta360),(1-mu)*sin(Theta360),zeros(1,length(Theta360)),'k--')
    plot3(rPh_syn(indPh_syn,1),rPh_syn(indPh_syn,2),zeros(indPh_syn,1),'k.','MarkerSize',50);
    plot3(rSC_syn(1:indPh_syn,1),rSC_syn(1:indPh_syn,2),zeros(indPh_syn,1),'b--');
    plot3(rSC_syn(indPh_syn,1),rSC_syn(indPh_syn,2),zeros(indPh_syn,1),'b.','MarkerSize',20);

    hold off

else

    figure('Renderer','painters','position',[100 100 1200 600])
    hold on
    grid minor
    axis equal
    title('Synodic Phobos flyby')
    xlabel(char(958));
    ylabel(char(951), 'Rotation', 360);

    theta_iniziale_P = data.th0P;

    % Disegno il pallino iniziale
    pallinoP = plot(xP_syn, yP_syn, 'k.', 'MarkerSize', 50);

    plot3((1-mu)*cos(Theta360),(1-mu)*sin(Theta360),zeros(1,length(Theta360)),'k--')                    % Plotto l'orbita di Phobos

    if indDe_syn < 6, indDe_syn = 6; end
    %if indDe_syn > length(t)-4, indDe_syn = length(t)-4; end

    % Animazione del movimento
    for t = indPh_syn - 5 : 1 : indPh_syn + 5                                                           % Plotto i passi temporali immediatamente prima e dopo il flyby

        % Calcolo le nuove coordinate del pallino
        % theta = theta_iniziale + theta;
        xP_syn = (1-mu) * cos(theta_iniziale_P);
        yP_syn = (1-mu) * sin(theta_iniziale_P);

        rPh_syn(t,:) = [xP_syn, yP_syn];
        rSC_syn(t,:) = [x_syn(t),y_syn(t)];

        % Aggiorno la posizione del pallino
        set(pallinoP, 'XData', xP_syn, 'YData', yP_syn);

        % Aggiorno i limiti degli assi in base alla posizione corrente del pallino
        xlim([xP_syn-.5 xP_syn+.5]);
        ylim([yP_syn-.5 yP_syn+.5]);

        hp = plot3(x_syn(1:t),y_syn(1:t),z_syn(1:t),'r-');

        % Aggiorno la figura
        pause(1)
        %delete(hp)
    end

    hold off

end

%% FLYBY DI DEIMOS SINODICO

if anim == 1

    figure('Renderer','painters','position',[100 100 1200 600])
    hold on
    grid minor
    axis equal
    title('Synodic Deimos flyby')
    xlabel(char(958));
    ylabel(char(951), 'Rotation', 360);

    xlim([rDe_syn(indDe_syn,1)-.5 rDe_syn(indDe_syn,1)+.5]);
    ylim([rDe_syn(indDe_syn,2)-.5 rDe_syn(indDe_syn,2)+.5]);

    plot3(data.DU_13_adim*cos(Theta360),data.DU_13_adim*sin(Theta360),zeros(1,length(Theta360)),'g--')
    plot3(rDe_syn(indDe_syn,1),rDe_syn(indDe_syn,2),zeros(indDe_syn,1),'g.','MarkerSize',50);
    plot3(rSC_syn(1:indDe_syn,1),rSC_syn(1:indDe_syn,2),zeros(indDe_syn,1),'b--');
    plot3(rSC_syn(indDe_syn,1),rSC_syn(indDe_syn,2),zeros(indDe_syn,1),'b.','MarkerSize',20);

    hold off

else

    figure('Renderer','painters','position',[100 100 1200 600])
    hold on
    grid minor
    axis equal
    title('Synodic Deimos flyby')
    xlabel(char(958));
    ylabel(char(951), 'Rotation', 360);

    theta_iniziale_D = data.th0D;

    % Disegno il pallino iniziale
    pallinoD = plot(xD_syn, yD_syn, 'g.', 'MarkerSize', 50);

    plot3(data.DU_13_adim*cos(Theta360),data.DU_13_adim*sin(Theta360),zeros(1,length(Theta360)),'g--')                    % Plotto l'orbita di Phobos

    % Animazione del movimento
    for t = indDe_syn - 5 : 1 : indDe_syn + 5                                                                             % Plotto i passi temporali immediatamente prima e dopo il flyby

        % Calcolo le nuove coordinate del pallino
        % theta = theta_iniziale + theta;
        xD_syn = data.DU_13_adim * cos( (data.omega_3/data.omega_2 - 1) *tout(t) + (theta_iniziale_D-theta_iniziale_P));
        yD_syn = data.DU_13_adim * sin( (data.omega_3/data.omega_2 - 1) *tout(t) + (theta_iniziale_D-theta_iniziale_P));

        rDe_syn(t,:) = [xD_syn, yD_syn];
        rSC_syn(t,:) = [x_syn(t),y_syn(t)];

        % Aggiorno la posizione del pallino
        set(pallinoD, 'XData', xD_syn, 'YData', yD_syn);

        % Aggiorno i limiti degli assi in base alla posizione corrente del pallino
        xlim([xD_syn-.5 xD_syn+.5]);
        ylim([yD_syn-.5 yD_syn+.5]);

        hp = plot3(x_syn(1:t),y_syn(1:t),z_syn(1:t),'r-');

        % Aggiorno la figura
        pause(1) %1e-2
        %delete(hp)
    end

    hold off

end

% subplot(2,2,1);
% subplot(2,2,2);
% subplot(2,2,3);
% subplot(2,2,4);
% exportgraphics(gcf, 'Sinodico_flyby_Phobos_theta300.eps', 'Resolution', 300);

%% PLOT DELL'INTERA TRAIETTORIA SINODICO

figure('Renderer','painters','position',[100 100 1200 600])
hold on
grid minor
axis equal
xlabel(char(958));
ylabel(char(951), 'Rotation', 360);

rectangle('Position', [xCenter - raggio, yCenter - raggio, 2*raggio, 2*raggio],...
    'Curvature', [1, 1], 'FaceColor', 'r')
plot3(-mu,0,0,'r.','MarkerSize',50)                                                   % Plotto Marte in posizione -mu nel sistema sinodico

% plot3(rPh_syn(end,1),rPh_syn(end,2),0,'k.','MarkerSize',10)
% plot3(rDe_syn(end,1),rDe_syn(end,2),0,'g.','MarkerSize',10)

plot3(rPh_syn(indPh_syn,1),rPh_syn(indPh_syn,2),0,'k.','MarkerSize',20)               % Plotto Phobos nella sua posizione nel momento del flyby nel sistema sinodico
plot3(rDe_syn(indDe_syn,1),rDe_syn(indDe_syn,2),0,'g.','MarkerSize',20)               % Plotto Deimos nella sua posizione nel momento del flyby nel sistema sinodico

%plot3(1-mu,0,0,'k.','MarkerSize',20)                                                 % Plotto Phobos in posizione 1-mu nel sistema sinodico
%plot3(data.DU_13_adim,0,0,'g.','MarkerSize',20)                                   % Plotto Deimos in posizione DU_13_adim

plot3(-mu*cos(Theta360),-mu*sin(Theta360),zeros(1,length(Theta360)),'r--')            % Plotto l'orbita di Marte
plot3((1-mu)*cos(Theta360),(1-mu)*sin(Theta360),zeros(1,length(Theta360)),'k--')      % Plotto l'orbita di Phobos
plot3(data.DU_13_adim*cos(Theta360),data.DU_13_adim*sin(Theta360),zeros(1,length(Theta360)),'g--')  % Plotto l'orbita di Deimos

plot3((altitudine/distanza)*cos(Theta360),(altitudine/distanza)*sin(Theta360), ...
    zeros(1,length(Theta360)),'b--')                                                  % Plotto l'orbita iniziale dello S/C

if norm(r_target) > 30
    plot3((Raggio_influenza_Marte/distanza)*cos(Theta360), ...
        (Raggio_influenza_Marte/distanza)*sin(Theta360), zeros(1,length(Theta360)),'y.-') % Plotto la sfera di influenza di Marte
end

plot3(0,0,0,'y.')                                                                 % Plotto il CM del sinodico
plot3(yout(1,1),yout(1,2),0,'b*')                                                 % Punto di partenza S\C
plot3(yout(end,1),yout(end,2),0,'c*')                                             % Punto finale S\C
%view(0,90)

if anim == 1
    plot3(x_syn,y_syn,z_syn)                                                      % Plotto la traiettoria
else
    comet3(x_syn,y_syn,z_syn)                                                     % Plotto la traiettoria
end

title('Mars-Phobos-Deimos 2D Synodic System')
xlabel(char(958)); ylabel(char(951), 'Rotation', 360);
legend('Mars','Phobos','Deimos','Mars orbit','Phobos orbit','Deimos orbit', ...
    'Initial orbit S/C','SOI Mars','CM Inertial','Starting point','Final point','Trajectory') %SOI Marte',

% Creazione della legenda
lgd = legend('Location', 'northwest');
numColumns = 2;  % Numero di colonne desiderate nella legenda
lgd.NumColumns = numColumns;

% set(gca, 'View', [-90 90]); % Ruota la visualizzazione
% axis([-62, 62, -120, 120]);

%     set(gca, 'fontsize', 14)
%     set(h,'fontsize',14)

hold off

%% PLOT VELOCITA SINODICO

figure('Renderer','painters','position',[100 100 1200 600])
hold on
grid minor
plot(vx_syn);
plot(vy_syn);
%plot(vz_in);
v_syn_mod = sqrt(vx_syn.^2 + vy_syn.^2);
plot(v_syn_mod,'LineWidth',2);
xlabel('time steps')
ylabel('Velocity');
yline(Vel_Inf/vel_unit);
%xline(indPh_in,'Color', 'green');
%xline(indDe_in,'Color', 'green');
legend('VX\_syn','VY\_syn','V\_syn\_mod','V\infty')%,'minPh','minDe')
title('Synodic velocities')
hold off

%% CONFRONTO VELOCITA INERZIALE VS SINODICO

figure('Renderer','painters','position',[100 100 1200 600])
grid minor
xlabel('time steps')
ylabel('Velocity');
yyaxis left
hold on
plot(v_syn_mod,'LineWidth',2);
[v_syn_mod_max,ind_syn_max] = max(v_syn_mod);
plot(ind_syn_max,v_syn_mod_max,'b*')
yyaxis right
hold on
plot(v_in_mod,'LineWidth',2);
[v_in_mod_max,ind_in_max] = min(v_in_mod);
plot(ind_in_max,v_in_mod_max,'r*')
legend('SYNODIC','max\_syn','INERTIAL','min\_inerz')
hold off

%% CONTROLLO POSIZIONE DI PARTENZA E ARRIVO

r1 = altitudine/distanza;                                                   % raggiovettore iniziale desiderato
r2 = norm(yout(1,1:2),2);                                                   % raggiovettore iniziale

if(abs(r1-r2)<(Raggio_Phobos/distanza))
    disp('Constraint on starting position respected')
else
    disp('Constraint on starting position NOT respected')
end


r3 = Raggio_influenza_Marte/distanza;                                       % raggiovettore finale desiderato
r4 = norm(yout(end,1:2),2);                                                 % raggiovettore finale

if(abs(r3-r4)<1e-2)
    disp('Constraint on final position respected')
else
    disp('Constraint on final position NOT respected')
end

disp('The S/C arrives at the Mars SOI with a vel_inf in [km/s] of')
disp( sqrt( vx_in(end)^2 + vy_in(end)^2 ) * vel_unit )
disp('spending a DV in [km/s] of')
disp( xf(1) )

%% CONTROLLO ENERGIA

V = sqrt(vx_in.^2 + vy_in.^2) .* vel_unit;
r = sqrt(x_in.^2 + y_in.^2) .* distanza;
En = V.^2./2 - data.mu1_dim./r;                                          % Energia dell'orbita, è una costante al netto di perturbazioni e interazioni con corpi secondari/terziari

figure('Renderer','painters','position',[100 100 1200 600])
hold on
grid minor
% title('Orbit energy')
ylabel('Orbit energy')
xlabel('time steps')
plot(En,'LineWidth',2)
xline(indPh_in,'Color', 'green');
xline(indDe_in,'Color', 'green');
h=legend('S/C trajectory energy','Minimum from Phobos','Minimum from Deimos');

set(gca, 'fontsize', 14)
set(h,'fontsize',14)

disp('with an energy equal to')
disp(En(end))



%% CONTROLLO DISTANZE RELATIVE

t_dim = linspace(0,48.3139,length(tout));

figure('Renderer','painters','position',[100 100 1200 600])
hold on
grid minor
title('Relative distances of the S/C from Phobos and Deimos')
xlabel('Dimensionless time')
ylabel('Dimensionless distance'); %csi
plot(tout,rSCPh_in,'Linewidth',2)
plot(tout(indPh_in),mPh_in,'b*')
plot(tout,rSCDe_in,'Linewidth',2)
plot(tout(indDe_in),mDe_in,'r*')
% plot(rSCPh_syn)
% plot(rSCDe_syn)
h = legend('r\_SC-Ph Inerz','Min from Phobos','r\_SC-De Inerz', 'Min from Deimos');
set(gca, 'fontsize', 14)
set(h,'fontsize',14)
hold off

%% PLOT DELTA THETA IN UN ANNO

t_year_dim = 3.156e7;                                                    % Numero di secondi in un anno
t_linspace = linspace(0, 365*86400, 1800);                               % 86400

% Calcolo della posizione angolare di Phobos in gradi
theta_Phobos = (t_linspace .* data.omega_2) .* 180 / pi;
theta_Deimos = (t_linspace .* data.omega_3) .* 180 / pi;

% Creazione del grafico
figure('Renderer','painters','position',[100 100 1200 600])
hold on
grid minor
xlabel('Time in a year [s]');
ylabel('Angular position [deg]');
title('Variation of Th0D - Th0P in one year');
yline(0)

plot(t_linspace, wrapTo180(theta_Deimos - theta_Phobos))                 % wrapTo180 = avvolgi gli angoli da -180 a 180 e ricomincia
dt = wrapTo180(theta_Deimos - theta_Phobos);

conteggio_intersezioni = 0;

% Verifica dell'attraversamento della retta a 0 gradi
for i = 2:length(t_linspace)
    if dt(i-1)<0 && dt(i) >0
        %     if (theta_Deimos(i) - theta_Phobos(i)) * (theta_Deimos(i - 1) - theta_Phobos(i - 1)) < 0
        % se la differenza tra le posizioni angolari di Deimos e Phobos cambia di segno
        % tra due punti consecutivi allora incremento il contatore
        conteggio_intersezioni = conteggio_intersezioni + 1;
    end
end
set(gca, 'fontsize', 14)
disp(['The number of intersections with the line at 0 degrees is: ' num2str(conteggio_intersezioni)]);


%% PLOT DV - MU

% DeltaV = [1.2036, 1.2036, 1.2036, 1.2010, 1.1888, 1.0352, 0.6711];
% Mu = [1.6606e-8, 1.6606e-7, 1.6606e-7, 1.6606e-5, 1.6606e-4, 1.6606e-3, 1.6606e-2];
%
% figure('Renderer','painters','position',[100 100 1200 600])
% hold on
% grid minor
% xlabel('mu [/]');
% ylabel('DeltaV [km/s]');
% title('DV vs mu');
% plot(Mu,DeltaV)



%% EFFEMEIDI

[position, velocity] = planetEphemeris(juliandate(1990,12,1),'SolarSystem','Mars');

%target — Target body (astronomical object) or point of reference
%'Sun' | 'Mercury' | 'Venus' | 'Earth' | 'Moon' | 'Mars' | 'Jupiter' | 'Saturn' | 'Uranus' | 'Neptune' | 'Pluto' | 'SolarSystem' | 'EarthMoon'






